#! /usr/bin/env python
"""Main script

This program is a part of Uptime package.

"""

__author__ = "Oleg Broytman <phd@phdru.name>"
__copyright__ = "Copyright (C) 2003-2013 PhiloSoft Design"
__license__ = "GNU GPL"


import sys, os
from tests import Test, StopTest


home_dir = os.path.dirname(os.path.abspath(sys.argv[0]))
me = os.path.basename(sys.argv[0])


tests_classes = {}


def import_tests():
   import os
   files = filter(lambda name: name.startswith("test_") and name.endswith(".py"),
      os.listdir(os.path.join(home_dir, "tests")))
   for file in files:
      module_name = file[:-3] # remove .py
      module = __import__("tests", globals(), locals(), [module_name])
      module = getattr(module, module_name)
      module_list = dir(module)
      for name in module_list:
         if name <> "Test":
            obj = getattr(module, name)
            try:
               if issubclass(obj, Test):
                  tests_classes[name] = obj
            except TypeError:
               pass

   if not tests_classes:
      raise RuntimeError, "no test modules found"


def load_test(config, section):
   if not tests_classes: import_tests()

   if not config.has_section(section):
      raise ConfigParser.NoSectionError, section

   kw = {}
   for option in config.options(section):
      kw[option] = config.get(section, option)

   klass = tests_classes[kw["class"]]
   del kw["class"]

   test = klass(**kw)
   return test


def run(verbose, real_run):
   if verbose: print "Parsing config..."
   import ConfigParser
   config = ConfigParser.ConfigParser()
   config.optionxform = str # make options' names case-sensitive
   config.read("uptime.conf")

   if not config.has_section("tests"):
      raise ConfigParser.NoSectionError, "tests"

   tests_num = config.getint("tests", "tests")
   if verbose: print "Ok.", tests_num, "tests."


   from quixote import enable_ptl
   enable_ptl() # enable it early to use templates to display tracebacks


   results = []
   if real_run:
      if verbose: print "Running..."
      try:
         for test_no in range(1, tests_num+1):
            test_id = "test%d" % test_no
            if verbose: print "   ", test_id

            if not config.has_option("tests", test_id):
               raise ConfigParser.NoSectionError, "test%d" % test_id

            test = load_test(config, config.get("tests", test_id))
            test.run()
            if test.result: results.append(test.result)
            if verbose: print "        finished"

      except StopTest:
         if verbose: print "    interrupted"
         if test.result: results.append(test.result)


   if config.has_option("uptime", "title"):
      title = config.get("uptime", "title")
   else:
      title = "Uptime"

   if config.has_option("uptime", "store"):
      store = config.getint("uptime", "store")
   else:
      store = 100

   if config.has_option("uptime", "page_size"):
      page_size = config.getint("uptime", "page_size")
   else:
      page_size = 25


   import time
   timestamp = time.time()

   import shelve
   db = shelve.open("uptime.db", 'c')

   if results:
      if verbose: print "Writing results to DB..."
      db[str(timestamp)] = results

   if verbose: print "Updating DB..."
   keys = db.keys()
   keys.sort()
   keys.reverse()

   for key in keys[store:]:
      del db[key]
   keys = keys[:store]

   if verbose: print "Building HTML pages..."
   pages = []
   start_page = 0
   save_keys = keys[:]

   while keys:
      old_keys = keys[:page_size]
      page = (start_page, start_page + len(old_keys))
      start_page += len(old_keys)
      pages.append(page)
      keys = keys[page_size:]

   keys = save_keys


   import locale
   locale.setlocale(locale.LC_TIME, '')
   dt_format = locale.nl_langinfo(locale.D_T_FMT)


   if verbose: print "    index"
   from templates.html import index
   html = str(index(title))
   index_html = file("public_html/index.html", 'w')
   index_html.write(html)
   index_html.close()

   from templates.db2html import db2html
   if len(pages) > 1:
      for p in range(len(pages)-1):
         if verbose: print "    page", p+1
         html = str(db2html(db, keys, title, p, pages, True, dt_format))
         page_html = file("public_html/page%d.html" % (p+1), 'w')
         page_html.write(html)
         page_html.close()

      p = len(pages)-1

   else:
      p = 0

   if verbose: print "    page", p+1
   html = str(db2html(db, keys, title, p, pages, False, dt_format))
   page_html = file("public_html/page%d.html" % (p+1), 'w')
   page_html.write(html)
   page_html.close()


   db.close()
   if verbose: print "Stop!"


def version(exit=1):
   from setup import _version
   sys.stdout.write("""\
Broytman uptime.py version %s, %s
""" % (_version, __copyright__))
   if exit: sys.exit(0)


def usage(code=0):
   version(0)
   sys.stdout.write("""\
Usage: %s [-h|--help] [-V|--version] [-v|--verbose] [-n|--dry-run]
""" % me)
   sys.exit(code)


def get_args():
   from getopt import getopt, GetoptError

   try:
      options, arguments = getopt(sys.argv[1:], "hVvn",
         ["help", "version", "verbose", "dry-run"])
   except GetoptError:
      usage(1)

   verbose = False
   real_run = True

   for option, value in options:
      if option == '-h':
         usage()
      elif option == '--help':
         usage()
      elif option == '-V':
         version()
      elif option == '--version':
         version()
      elif option == '-v':
         verbose = True
      elif option == '--verbose':
         verbose = True
      elif option == '-n':
         real_run = False
      elif option == '--dry-run':
         real_run = False
      else:
         usage(1)

   if arguments: usage(1)
   return verbose, real_run


if __name__ == "__main__":
   verbose, real_run = get_args()
   run(verbose, real_run)
